%% Investigating the American Dream: The Role of Neighborhoods

% This master file runs all the main exercises by calling the relevant
% programs

%% Static Exercises

clear 
close all

global alpha beta b H a N_a eta log_a_mu  mp N_w

% Parameters from calibration
x_s = [1.6466    1.1994    0.6920    0.1328    1.2834    0.3609]; 

H     = .1935;  % Housing supply
b     = x_s(1); % Wage Function parameter
beta  = x_s(2); % Wage function parameter
alpha = x_s(3); % Wage function parameter
omega = x_s(4); % Spillover parameter
sig   = x_s(5); % St Dev log ability
rho   = x_s(6); % Autocorrelation of log ability

% Parameters from Steady State

S_A   = 1.0675; % Spillover A    
S_B   = 0.5898; % Spillover VB

% Externally set parameters
log_w_sigma = 0.5; % St Dev log wage  
log_w_mu    = 1;   % Mean log wage

corr_wa     = 0;

% Normalization
log_a_mu = -3.3;

% Grid size
mp   = 4;

param = [ omega,sig, rho, S_A, S_B,log_w_sigma,corr_wa,log_w_mu,1];

%% Comparative statics: Varying Parental Income’s Volatility

%  Grids

N_w = 200;
N_a = 200;

T     = 1; % Time (one period = static exercise)

sigma_grid = linspace(0.1,0.5,5); % Grid for income volatility

dissim_t      = zeros(length(sigma_grid),1);
r_A_t         = zeros(length(sigma_grid),1);
Q1toQ1_t      = zeros(length(sigma_grid),1);
Q1toQ4_t      = zeros(length(sigma_grid),1);
Q4toQ4_t      = zeros(length(sigma_grid),1);
rank_corr_t   = zeros(length(sigma_grid),1);
dissim_ab_t   = zeros(length(sigma_grid),1);

% Solve equilibrium and store results
for i=1:length(sigma_grid)
    
    log_w_sigma = sigma_grid(i);
    param = [omega,sig, rho, S_A, S_B,log_w_sigma,corr_wa,log_w_mu,T];
    
    [TimeSeries] = hand_fixS(param); % Solve the equilibrium

    dissim_t(i)      = cell2mat(TimeSeries(1,1));
    r_A_t(i)         = cell2mat(TimeSeries(1,11));
    Q1toQ1_t(i)      = cell2mat(TimeSeries(1,4));
    Q1toQ4_t(i)      = cell2mat(TimeSeries(1,5));
    Q4toQ4_t(i)      = cell2mat(TimeSeries(1,6));
    rank_corr_t(i)   = cell2mat(TimeSeries(1,7));
    dissim_ab_t(i)     = cell2mat(TimeSeries(1,27));
end

excess_demand_t = zeros(1,length(sigma_grid));
for i=1:length(sigma_grid)
    
    log_w_sigma = sigma_grid(i);
    param = [omega,sig, rho, S_A, S_B,log_w_sigma,corr_wa,log_w_mu,T];

    [exc_dem] = excess_demand(param,r_A_t(1));

    excess_demand_t(i)      = exc_dem;
end


%% Figures

fig=figure;
hold on
plot(sigma_grid, dissim_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Dissimilarity by Income')
xlabel('Variance of Parents log Wage')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-change-varlogw-diss.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(sigma_grid, dissim_ab_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Dissimilarity by Productivity')
xlabel('Variance of Parents log Wage')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-change-varlogw-dissim-ab.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(sigma_grid, r_A_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Rent A')
xlabel('Variance of Parents log Wage')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-change-varlogw-rent.pdf','-dpdf','-r0')


fig=figure;
hold on
plot(sigma_grid, excess_demand_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Excess Demand')
xlabel('Variance of Parents log Wage')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-change-varlogw-excess-demand.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(sigma_grid, rank_corr_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Rank-Rank')
xlabel('Variance of Parents log Wage')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-change-varlogw-rankrank.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(sigma_grid, Q1toQ1_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(sigma_grid, Q1toQ4_t,"--",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(sigma_grid, Q4toQ4_t,":",'LineWidth',2,'Color', [0    0.4470    0.7410] )
ylim([.15 .4])
lgd=legend({'Q1 to Q1','Q1 to Q4','Q4 to Q4'}, 'Location','north','FontSize',12,'Box','off');
lgd.NumColumns = 3;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Quartile Transitions')
xlabel('Variance of Parents log Wage')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-change-varlogw-Qtrans.pdf','-dpdf','-r0')




%% Comparative statics: Varying the Spillover Gap

% Grids
N_w = 100;
N_a = 100;

log_w_sigma = 0.5;  
S_A_grid = linspace(S_B*1.2,S_B*6,10);

T=1;

dissim_t        = zeros(length(S_A_grid),1);
r_A_t           = zeros(length(S_A_grid),1);
Q1toQ1_t        = zeros(length(S_A_grid),1);
Q1toQ4_t        = zeros(length(S_A_grid),1);
Q4toQ4_t        = zeros(length(S_A_grid),1);
rank_corr_t     = zeros(length(S_A_grid),1);
var_logw_kids_t = zeros(length(S_A_grid),1);
dissim_ab_t     = zeros(length(S_A_grid),1);

for i=1:length(S_A_grid)
    
    param = [omega,sig, rho, S_A_grid(i), S_B,log_w_sigma,corr_wa,log_w_mu,T];

    [TimeSeries] = hand_fixS(param);

    dissim_t(i)      = cell2mat(TimeSeries(1,1));
    r_A_t(i)         = cell2mat(TimeSeries(1,11));
    Q1toQ1_t(i)      = cell2mat(TimeSeries(1,4));
    Q1toQ4_t(i)      = cell2mat(TimeSeries(1,5));
    Q4toQ4_t(i)      = cell2mat(TimeSeries(1,6));
    rank_corr_t(i)   = cell2mat(TimeSeries(1,7));
    var_logw_kids_t(i)  = cell2mat(TimeSeries(1,26));
    dissim_ab_t(i)      = cell2mat(TimeSeries(1,27));
 end

excess_demand_t = zeros(1,length(S_A_grid));
for i=1:length(S_A_grid)
    
    param = [omega,sig, rho, S_A_grid(i), S_B,log_w_sigma,corr_wa,log_w_mu,T];

    [exc_dem] = excess_demand(param,r_A_t(1));

    excess_demand_t(i)      = exc_dem;
end


spill_gap = [S_A_grid-S_B]';

%% Figures

fig=figure;
hold on
plot(spill_gap, dissim_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(spill_gap, dissim_ab_t,"--",'LineWidth',2,'Color', [0    0.4470    0.7410] )
xlim([round(spill_gap(1),1),round(spill_gap(end),1)])
ylim([0.25 0.7])
lgd=legend({'by Income','by Productivity'}, 'Location','northeast','FontSize',11,'Box','off');
lgd.NumColumns = 2;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Dissimilarity')
xlabel('Spillover Gap')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-spillgap-diss-inc-ab.pdf','-dpdf','-r0')


fig=figure;
hold on
plot(spill_gap, var_logw_kids_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
xlim([round(spill_gap(1),1),round(spill_gap(end),1)])
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Variance of Log Children Income')
xlabel('Spillover Gap')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-spillgap-varlogw-kids.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(spill_gap, r_A_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
xlim([round(spill_gap(1),1),round(spill_gap(end),1)])
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Rent A')
xlabel('Spillover Gap')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-spillgap-rent.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(spill_gap, excess_demand_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
xlim([round(spill_gap(1),1),round(spill_gap(end),1)])
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Excess Demand')
xlabel('Spillover Gap')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-spillgap-excess-demand.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(spill_gap, rank_corr_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
xlim([round(spill_gap(1),1),round(spill_gap(end),1)])
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Rank Rank')
xlabel('Spillover Gap')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-spillgap-rankrank.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(spill_gap, Q1toQ1_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(spill_gap, Q1toQ4_t,"--",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(spill_gap, Q4toQ4_t,":",'LineWidth',2,'Color', [0    0.4470    0.7410] )
ylim([0.05,0.55])
xlim([round(spill_gap(1),1),round(spill_gap(end),1)])
lgd=legend({'Q1 to Q1','Q1 to Q4','Q4 to Q4'}, 'Location','north','FontSize',12,'Box','off');
lgd.NumColumns = 3;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Quartile Transitions')
xlabel('Spillover Gap')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-spillgap-Qtrans.pdf','-dpdf','-r0')


%% Planner: compare to equilibrium varying spillover gap

N_a = 100;
N_w = 100;

log_w_sigma = 0.5;  
corr_wa = 0;

S_A_grid = linspace(S_B*1.2,S_B*6,10);

T=1;

dissim_t        = zeros(length(S_A_grid),1);
avg_ut_t        = zeros(length(S_A_grid),1);
var_logw_kids_t = zeros(length(S_A_grid),1);
var_log_cons_t  = zeros(length(S_A_grid),1);
Q1toQ1_t        = zeros(length(S_A_grid),1);
Q1toQ4_t        = zeros(length(S_A_grid),1);
Q4toQ4_t        = zeros(length(S_A_grid),1);
rank_corr_t     = zeros(length(S_A_grid),1);

dissim_p        = zeros(length(S_A_grid),1);
avg_ut_p        = zeros(length(S_A_grid),1);
var_logw_kids_p = zeros(length(S_A_grid),1);
var_log_cons_p  = zeros(length(S_A_grid),1);
Q1toQ1_p        = zeros(length(S_A_grid),1);
Q1toQ4_p        = zeros(length(S_A_grid),1);
Q4toQ4_p        = zeros(length(S_A_grid),1);
rank_corr_p     = zeros(length(S_A_grid),1);


for i=1:length(S_A_grid)
    param = [omega,sig, rho, S_A_grid(i), S_B,log_w_sigma,corr_wa,log_w_mu,T];

    [TimeSeries] = hand_fixS(param);

    
    dissim_t(i)         = cell2mat(TimeSeries(1,1));
    Q1toQ1_t(i)         = cell2mat(TimeSeries(1,4));
    Q1toQ4_t(i)         = cell2mat(TimeSeries(1,5));
    Q4toQ4_t(i)         = cell2mat(TimeSeries(1,6));
    rank_corr_t(i)      = cell2mat(TimeSeries(1,7));
    avg_ut_t(i)         = cell2mat(TimeSeries(1,22));
    var_log_cons_t(i)   = cell2mat(TimeSeries(1,25));
    var_logw_kids_t(i)  = cell2mat(TimeSeries(1,26));

    [TimeSeries] = hand_fixS_planner(param);

    dissim_p(i)      = cell2mat(TimeSeries(1,1));
    Q1toQ1_p(i)        = cell2mat(TimeSeries(1,4));
    Q1toQ4_p(i)        = cell2mat(TimeSeries(1,5));
    Q4toQ4_p(i)        = cell2mat(TimeSeries(1,6));
    rank_corr_p(i)     = cell2mat(TimeSeries(1,7));
    avg_ut_p(i)        = cell2mat(TimeSeries(1,22));
    var_log_cons_p(i)  = cell2mat(TimeSeries(1,25));
    var_logw_kids_p(i)  = cell2mat(TimeSeries(1,26));
     
end

%% Figures
spill_gap = [S_A_grid-S_B]';


fig=figure;
hold on
plot(spill_gap, dissim_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(spill_gap, dissim_p(1:length(dissim_t)),"--",'LineWidth',2.5)
xlim([round(spill_gap(1),1),round(spill_gap(end),1)])
ylim([-0.05 0.5])
lgd=legend({'Equilibrium','Planner'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 2;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Dissimilarity')
xlabel('Spillover Gap')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-planner-spillgap-diss.pdf','-dpdf','-r0')


fig=figure;
hold on
plot(spill_gap, avg_ut_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(spill_gap, avg_ut_p(1:length(dissim_t)),"--",'LineWidth',2.5)
xlim([round(spill_gap(1),1),round(spill_gap(end),1)])
lgd=legend({'Equilibrium','Planner'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 2;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Welfare')
xlabel('Spillover Gap')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-planner-spillgap-welfare.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(spill_gap, var_log_cons_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(spill_gap, var_log_cons_p,"--",'LineWidth',2.5)
xlim([round(spill_gap(1),1),round(spill_gap(end),1)])
ylim([-0.01 0.3])
lgd=legend({'Equilibrium','Planner'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 2;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Variance of Log Consumption')
xlabel('Spillover Gap')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-planner-spillgap-varlogc.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(spill_gap, var_logw_kids_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(spill_gap, var_logw_kids_p,"--",'LineWidth',2.5)
xlim([round(spill_gap(1),1),round(spill_gap(end),1)])
lgd=legend({'Equilibrium','Planner'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 2;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Variance of Log Children Income')
xlabel('Spillover Gap')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-planner-spillgap-varlogkidsw.pdf','-dpdf','-r0')


fig=figure;
hold on
plot(spill_gap, rank_corr_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(spill_gap, rank_corr_p(1:length(dissim_t)),"--",'LineWidth',2.5)
xlim([round(spill_gap(1),1),round(spill_gap(end),1)])
lgd=legend({'Equilibrium','Planner'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 2;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Rank Rank')
xlabel('Spillover Gap')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-planner-spillgap-rankrank.pdf','-dpdf','-r0')

fig=figure;
hold on
h1=plot(spill_gap, Q1toQ1_t,"o-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
h2=plot(spill_gap, Q1toQ4_t,"o--",'LineWidth',2,'Color', [0    0.4470    0.7410] )
h3=plot(spill_gap, Q4toQ4_t,"o:",'LineWidth',2,'Color', [0    0.4470    0.7410] )
h4=plot(spill_gap, Q1toQ1_p(1:length(dissim_t)),"s-",'LineWidth',2.5,'Color', [0.8500    0.3250    0.0980])
h5=plot(spill_gap, Q1toQ4_p(1:length(dissim_t)),"s--",'LineWidth',2.5,'Color', [0.8500    0.3250    0.0980])
h6=plot(spill_gap, Q4toQ4_p(1:length(dissim_t)),"s:",'LineWidth',2.5,'Color', [0.8500    0.3250    0.0980])
xlim([round(spill_gap(1),1),round(spill_gap(end),1)])
lgd=legend( [h1 h4 h2 h5 h3 h6], ...
    {'Q1 to Q1 Equilibrium','Q1 to Q1 Planner','Q1 to Q4 Equilibrium','Q1 to Q4 Planner','Q4 to Q4 Equilibrium','Q4 to Q4 Planner'}, 'Location','northoutside','FontSize',9,'Box','off');
lgd.NumColumns = 3;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Quartile Transitions')
xlabel('Spillover Gap')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-planner-spillgap-Qtrans.pdf','-dpdf','-r0')

%% Planner: compare to equilibrium varying corr w-a

log_w_sigma = 0.5;  

N_a = 200;
N_w = 200;

corr_wa_grid = linspace(0,.99,10);
T = 1;

dissim_t        = zeros(length(S_A_grid),1);
avg_ut_t        = zeros(length(S_A_grid),1);
var_logw_kids_t = zeros(length(S_A_grid),1);
var_log_cons_t  = zeros(length(S_A_grid),1);
Q1toQ1_t        = zeros(length(S_A_grid),1);
Q1toQ4_t        = zeros(length(S_A_grid),1);
Q4toQ4_t        = zeros(length(S_A_grid),1);
rank_corr_t     = zeros(length(S_A_grid),1);

dissim_p        = zeros(length(S_A_grid),1);
avg_ut_p        = zeros(length(S_A_grid),1);
var_logw_kids_p = zeros(length(S_A_grid),1);
var_log_cons_p  = zeros(length(S_A_grid),1);
Q1toQ1_p        = zeros(length(S_A_grid),1);
Q1toQ4_p        = zeros(length(S_A_grid),1);
Q4toQ4_p        = zeros(length(S_A_grid),1);
rank_corr_p     = zeros(length(S_A_grid),1);


for i=1:length(corr_wa_grid)
    corr_wa = corr_wa_grid(i);
    param = [omega,sig, rho, S_A, S_B,log_w_sigma,corr_wa,log_w_mu,T];

    [TimeSeries] = hand_fixS(param);

    dissim_t(i)         = cell2mat(TimeSeries(1,1));
    Q1toQ1_t(i)         = cell2mat(TimeSeries(1,4));
    Q1toQ4_t(i)         = cell2mat(TimeSeries(1,5));
    Q4toQ4_t(i)         = cell2mat(TimeSeries(1,6));
    rank_corr_t(i)      = cell2mat(TimeSeries(1,7));
    avg_ut_t(i)        = cell2mat(TimeSeries(1,22));
    var_log_cons_t(i)  = cell2mat(TimeSeries(1,25));
    var_logw_kids_t(i)  = cell2mat(TimeSeries(1,26));
 
    [TimeSeries] = hand_fixS_planner(param);

    dissim_p(i)      = cell2mat(TimeSeries(1,1));
    Q1toQ1_p(i)        = cell2mat(TimeSeries(1,4));
    Q1toQ4_p(i)        = cell2mat(TimeSeries(1,5));
    Q4toQ4_p(i)        = cell2mat(TimeSeries(1,6));
    rank_corr_p(i)     = cell2mat(TimeSeries(1,7));
    avg_ut_p(i)        = cell2mat(TimeSeries(1,22));
    var_log_cons_p(i)  = cell2mat(TimeSeries(1,25));
    var_logw_kids_p(i)  = cell2mat(TimeSeries(1,26));
 
end


%% Figures
fig=figure;
hold on
plot(corr_wa_grid, dissim_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(corr_wa_grid, dissim_p,"--",'LineWidth',2.5)
lgd=legend({'Equilibrium','Planner'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 2;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Dissimilarity')
xlabel('Correlation Parent Wage and Child Ability')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-planner-corrwa-diss.pdf','-dpdf','-r0')



fig=figure;
hold on
plot(corr_wa_grid, avg_ut_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(corr_wa_grid, avg_ut_p,"--",'LineWidth',2.5)
lgd=legend({'Equilibrium','Planner'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 2;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Welfare')
xlabel('Correlation Parent Wage and Child Ability')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-planner-corrwa-welfare.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(corr_wa_grid, var_log_cons_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(corr_wa_grid, var_log_cons_p,"--",'LineWidth',2.5)
ylim([-0.01 0.3])
lgd=legend({'Equilibrium','Planner'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 2;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Variance of Log Consumption')
xlabel('Correlation Parent Wage and Child Ability')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-planner-corrwa-varlogc.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(corr_wa_grid, var_logw_kids_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(corr_wa_grid, var_logw_kids_p,"--",'LineWidth',2.5)
lgd=legend({'Equilibrium','Planner'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 2;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Variance of Log Children Income')
xlabel('Correlation Parent Wage and Child Ability')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-planner-corrwa-varlogkidsw.pdf','-dpdf','-r0')


fig=figure;
hold on
plot(corr_wa_grid, rank_corr_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(corr_wa_grid, rank_corr_p,"--",'LineWidth',2.5)
lgd=legend({'Equilibrium','Planner'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 2;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Rank Rank')
xlabel('Correlation Parent Wage and Child Ability')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-planner-corrwa-rankrank.pdf','-dpdf','-r0')

fig=figure;
hold on
h1=plot(corr_wa_grid, Q1toQ1_t,"o-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
h2=plot(corr_wa_grid, Q1toQ4_t,"o--",'LineWidth',2,'Color', [0    0.4470    0.7410] )
h3=plot(corr_wa_grid, Q4toQ4_t,"o:",'LineWidth',2,'Color', [0    0.4470    0.7410] )
h4=plot(corr_wa_grid, Q1toQ1_p,"s-",'LineWidth',2.5,'Color', [0.8500    0.3250    0.0980])
h5=plot(corr_wa_grid, Q1toQ4_p,"s--",'LineWidth',2.5,'Color', [0.8500    0.3250    0.0980])
h6=plot(corr_wa_grid, Q4toQ4_p,"s:",'LineWidth',2.5,'Color', [0.8500    0.3250    0.0980])
ylim([-0.05,1])
lgd=legend( [h1 h4 h2 h5 h3 h6], ...
    {'Q1 to Q1 Equilibrium','Q1 to Q1 Planner','Q1 to Q4 Equilibrium','Q1 to Q4 Planner','Q4 to Q4 Equilibrium','Q4 to Q4 Planner'}, 'Location','northoutside','FontSize',9,'Box','off');
lgd.NumColumns = 3;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Quartile Transitions')
xlabel('Correlation Parent Wage and Child Ability')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-planner-corrwa-Qtrans.pdf','-dpdf','-r0')



%% Policy

% A lump-sum transfer of an amount "tr" to the people in the lowest pth income
% percentile, financed with a proportional tax to people above the pth
% percentile 

N_a = 100;
N_w = 100;

log_w_sigma = 0.5;
corr_wa = 0;

S_A_grid = linspace(S_B*1.2,S_B*6,5);

T=1;

% Policy parameters
p = 0.25; % income percentile for eligibility
tr_rate = 0.2; % transfer as % of avg income

dissim_pol        = zeros(length(S_A_grid),1);
dissim_ab_pol     = zeros(length(S_A_grid),1);
avg_ut_pol        = zeros(length(S_A_grid),1);
var_logw_kids_pol = zeros(length(S_A_grid),1);
var_log_cons_pol  = zeros(length(S_A_grid),1);
Q1toQ1_pol        = zeros(length(S_A_grid),1);
Q1toQ4_pol        = zeros(length(S_A_grid),1);
Q4toQ4_pol        = zeros(length(S_A_grid),1);
rank_corr_pol     = zeros(length(S_A_grid),1);

dissim_t        = zeros(length(S_A_grid),1);
dissim_ab_t     = zeros(length(S_A_grid),1);
avg_ut_t        = zeros(length(S_A_grid),1);
var_logw_kids_t = zeros(length(S_A_grid),1);
var_log_cons_t  = zeros(length(S_A_grid),1);
Q1toQ1_t        = zeros(length(S_A_grid),1);
Q1toQ4_t        = zeros(length(S_A_grid),1);
Q4toQ4_t        = zeros(length(S_A_grid),1);
rank_corr_t     = zeros(length(S_A_grid),1);

dissim_p        = zeros(length(S_A_grid),1);
dissim_ab_p     = zeros(length(S_A_grid),1);
avg_ut_p        = zeros(length(S_A_grid),1);
var_logw_kids_p = zeros(length(S_A_grid),1);
var_log_cons_p  = zeros(length(S_A_grid),1);
Q1toQ1_p        = zeros(length(S_A_grid),1);
Q1toQ4_p        = zeros(length(S_A_grid),1);
Q4toQ4_p        = zeros(length(S_A_grid),1);
rank_corr_p     = zeros(length(S_A_grid),1);

for i=1:length(S_A_grid)
    param = [omega,sig, rho, S_A_grid(i), S_B,log_w_sigma,corr_wa,log_w_mu,T,p,tr_rate];

    [TimeSeries] = hand_fixS_policy_stat(param);
    dissim_pol(i)        = cell2mat(TimeSeries(1,1));
    Q1toQ1_pol(i)        = cell2mat(TimeSeries(1,4));
    Q1toQ4_pol(i)        = cell2mat(TimeSeries(1,5));
    Q4toQ4_pol(i)        = cell2mat(TimeSeries(1,6));
    rank_corr_pol(i)     = cell2mat(TimeSeries(1,7));
    avg_ut_pol(i)        = cell2mat(TimeSeries(1,22));
    var_log_cons_pol(i)  = cell2mat(TimeSeries(1,25));
    var_logw_kids_pol(i) = cell2mat(TimeSeries(1,26));
    dissim_ab_pol(i)     = cell2mat(TimeSeries(1,27));
     
    param = [omega,sig, rho, S_A_grid(i), S_B,log_w_sigma,corr_wa,log_w_mu,T];
    [TimeSeries] = hand_fixS(param);
    dissim_t(i)        = cell2mat(TimeSeries(1,1));
    Q1toQ1_t(i)        = cell2mat(TimeSeries(1,4));
    Q1toQ4_t(i)        = cell2mat(TimeSeries(1,5));
    Q4toQ4_t(i)        = cell2mat(TimeSeries(1,6));
    rank_corr_t(i)     = cell2mat(TimeSeries(1,7));
    avg_ut_t(i)        = cell2mat(TimeSeries(1,22));
    var_log_cons_t(i)  = cell2mat(TimeSeries(1,25));
    var_logw_kids_t(i) = cell2mat(TimeSeries(1,26));
    dissim_ab_t(i)     = cell2mat(TimeSeries(1,27));
   
    [TimeSeries] = hand_fixS_planner(param);
    dissim_p(i)      = cell2mat(TimeSeries(1,1));
    Q1toQ1_p(i)        = cell2mat(TimeSeries(1,4));
    Q1toQ4_p(i)        = cell2mat(TimeSeries(1,5));
    Q4toQ4_p(i)        = cell2mat(TimeSeries(1,6));
    rank_corr_p(i)     = cell2mat(TimeSeries(1,7));
    avg_ut_p(i)        = cell2mat(TimeSeries(1,22));
    var_log_cons_p(i)  = cell2mat(TimeSeries(1,25));
    var_logw_kids_p(i)  = cell2mat(TimeSeries(1,26));
    dissim_ab_p(i)      = cell2mat(TimeSeries(1,27));
    
end 


%% Figures
spill_gap = [S_A_grid-S_B]';


fig=figure;
hold on
plot(spill_gap, dissim_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(spill_gap, dissim_p,"--",'LineWidth',2.5)
plot(spill_gap, dissim_pol,":",'LineWidth',2.5,'color',[118,173,72]/255)
xlim([round(spill_gap(1),1),round(spill_gap(end),1)])
ylim([-0.01 0.5])
lgd=legend({'Equilibrium','Planner','Policy'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 3;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Dissimilarity')
xlabel('Spillover Gap')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-policy-spillgap-dissim.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(spill_gap, dissim_ab_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(spill_gap, dissim_ab_p,"--",'LineWidth',2.5)
plot(spill_gap, dissim_ab_pol,":",'LineWidth',2.5,'color',[118,173,72]/255)
ylim([0.4 1.05])
xlim([round(spill_gap(1),1),round(spill_gap(end),1)])
lgd=legend({'Equilibrium','Planner','Policy'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 3;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Dissimilarity by Productivity')
xlabel('Spillover Gap')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-policy-spillgap-diss-ab.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(spill_gap, avg_ut_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(spill_gap, avg_ut_p,"--",'LineWidth',2.5)
plot(spill_gap, avg_ut_pol,":",'LineWidth',2.5,'color',[118,173,72]/255)
xlim([round(spill_gap(1),1),round(spill_gap(end),1)])
lgd=legend({'Equilibrium','Planner','Policy'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 3;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Welfare')
xlabel('Spillover Gap')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-policy-spillgap-avgut.pdf','-dpdf','-r0')



fig=figure;
hold on
plot(spill_gap, var_log_cons_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(spill_gap, var_log_cons_p,"--",'LineWidth',2.5)
plot(spill_gap, var_log_cons_pol,":",'LineWidth',2.5,'color',[118,173,72]/255)
xlim([round(spill_gap(1),1),round(spill_gap(end),1)])
ylim([-0.01 0.3])
lgd=legend({'Equilibrium','Planner','Policy'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 3;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Variance of Log Consumption')
xlabel('Spillover Gap')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-policy-spillgap-varlogc.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(spill_gap, var_logw_kids_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(spill_gap, var_logw_kids_p,"--",'LineWidth',2.5)
plot(spill_gap, var_logw_kids_pol,":",'LineWidth',2.5,'color',[118,173,72]/255)
xlim([round(spill_gap(1),1),round(spill_gap(end),1)])
lgd=legend({'Equilibrium','Planner','Policy'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 3;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Variance of Log Children Income')
xlabel('Spillover Gap')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-policy-spillgap-varlogkidsw.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(spill_gap, rank_corr_t,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(spill_gap, rank_corr_p,"--",'LineWidth',2.5)
plot(spill_gap, rank_corr_pol,":",'LineWidth',2,'color',[118,173,72]/255)
xlim([round(spill_gap(1),1),round(spill_gap(end),1)])
ylim([-0.005 0.3])
lgd=legend({'Equilibrium','Planner','Policy'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 3;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Rank Rank')
xlabel('Spillover Gap')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-policy-spillgap-rankrank.pdf','-dpdf','-r0')

fig=figure;
hold on
h1=plot(spill_gap, Q1toQ1_t,"o-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
h4=plot(spill_gap, Q1toQ1_p,"o--",'LineWidth',2.5,'Color', [0.8500    0.3250    0.0980])
plot(spill_gap, Q1toQ1_pol,"o:",'LineWidth',2.5,'color',[118,173,72]/255)

h2=plot(spill_gap, Q1toQ4_t,"s-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
h5=plot(spill_gap, Q1toQ4_p(1:length(dissim_t)),"s--",'LineWidth',2.5,'Color', [0.8500    0.3250    0.0980])
plot(spill_gap, Q1toQ4_pol,"s:",'LineWidth',2.5,'color',[118,173,72]/255)

h3=plot(spill_gap, Q4toQ4_t,"^-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
h6=plot(spill_gap, Q4toQ4_p(1:length(dissim_t)),"^--",'LineWidth',2.5,'Color', [0.8500    0.3250    0.0980])
plot(spill_gap, Q4toQ4_pol,"^:",'LineWidth',2.5,'color',[118,173,72]/255)

xlim([round(spill_gap(1),1),round(spill_gap(end),1)])
lgd=legend( {'Q1 to Q1 Equilibrium','Q1 to Q1 Planner','Q1 to Q1 Policy','Q1 to Q4 Equilibrium','Q1 to Q4 Planner','Q1 to Q4 Policy','Q4 to Q4 Equilibrium','Q4 to Q4 Planner','Q4 to Q4 Policy'}, 'Location','northoutside','FontSize',9,'Box','off');
lgd.NumColumns = 3;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Quartile Transitions')
xlabel('Spillover Gap')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-ploicy-spillgap-Qtrans.pdf','-dpdf','-r0')

%% Endogenous spillovers: compare steady states changing omega
clear 
close all

global  H a N_a eta log_a_mu log_w_mu log_w_sigma log_e_mu

% Parameters
% Externally set and initial conditions
log_a_mu = -3.3;
log_w_mu = 1;
log_e_mu = .4;
log_w_sigma = 0.5;

N_a = 200;
N_w = 200;
N_n = 2;
mp = 4;

% Internally Calibrated
H = .1935;

x_s = [1.6466    1.1994    0.6920    0.1328    1.2834    0.3609    0.3410    0.3228];

omega = x_s(4);
omega_grid = linspace(0,1,5);

dissim_end        = zeros(length(omega_grid),1);
dissim_ab_end     = zeros(length(omega_grid),1);
var_logw_end      = zeros(length(omega_grid),1);
Q1toQ1_end        = zeros(length(omega_grid),1);
Q1toQ4_end        = zeros(length(omega_grid),1);
Q4toQ4_end        = zeros(length(omega_grid),1);
rank_corr_end     = zeros(length(omega_grid),1);
S_A_end           = zeros(length(omega_grid),1);
S_B_end           = zeros(length(omega_grid),1);


for i =1:length(omega_grid)
    
    [timeseries] = hand_endS_n_llprof([1.6466    1    0.6920    omega_grid(i) 1.2834    0.3609    0.3410    0.3228 ]);

    dissim_end(i)      = cell2mat(timeseries(end,1));
    Q1toQ1_end(i)      = cell2mat(timeseries(end,4));
    Q1toQ4_end(i)      = cell2mat(timeseries(end,5));
    Q4toQ4_end(i)      = cell2mat(timeseries(end,6));
    rank_corr_end(i)   = cell2mat(timeseries(end,7));
    S_A_end(i)         = cell2mat(timeseries(end,9));
    S_B_end(i)         = cell2mat(timeseries(end,10));
    var_logw_end(i)    = cell2mat(timeseries(end,19));
    dissim_ab_end(i)   = cell2mat(timeseries(end,26));

end


%% Figures 

fig=figure;
hold on
plot(omega_grid, dissim_end,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Dissimilarity by Income')
xlabel('Omega')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-changeomega-diss.pdf','-dpdf','-r0')


fig=figure;
hold on
plot(omega_grid, var_logw_end,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Variance of Log Income')
xlabel('Omega')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-changeomega-varlogw.pdf','-dpdf','-r0')


fig=figure;
hold on
plot(omega_grid, dissim_ab_end,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Dissimilarity by Productivity')
xlabel('Omega')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-changeomega-dissim-ab.pdf','-dpdf','-r0')


fig=figure;
hold on
plot(omega_grid, S_A_end-S_B_end,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Spillover Gap')
xlabel('Omega')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-changeomega-spillover-gap.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(omega_grid, rank_corr_end,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Rank Rank')
xlabel('Omega')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-changeomega-rankrank.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(omega_grid, Q1toQ1_end,"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(omega_grid, Q1toQ4_end,"--",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(omega_grid, Q4toQ4_end,":",'LineWidth',2,'Color', [0    0.4470    0.7410] )
lgd=legend({'Q1 to Q1','Q1 to Q4','Q4 to Q4'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 3;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Quartile Transitions')
xlabel('Omega')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-changeomega-Qtrans.pdf','-dpdf','-r0')


%% Skill Premium Shock

clear 
close all

global  H a N_a eta log_a_mu log_w_mu log_w_sigma log_e_mu

% Parameters
% Externally set and initial conditions
log_a_mu = -3.3;
log_w_mu = 1;
log_e_mu = .4;
log_w_sigma = 0.5;


% Internally Calibrated
H = .1935;
eta= 2.5; % Skill premium shock

x_s = [1.6466    1.1994    0.6920    0.1328    1.2834    0.3609    0.3410    0.3228 eta];


% Model with endogenous spillovers

[timeseries,t_ss] = hand_endS_n_llprof_eta(x_s);

dissim_end       = cell2mat(timeseries(:,1));
Q1toQ1_end       = cell2mat(timeseries(:,4));
Q1toQ4_end       = cell2mat(timeseries(:,5));
Q4toQ4_end       = cell2mat(timeseries(:,6));
rank_corr_end    = cell2mat(timeseries(:,7));
S_A_end          = cell2mat(timeseries(:,9));
S_B_end          = cell2mat(timeseries(:,10));
var_logw_k_end   = cell2mat(timeseries(:,19));
dissim_ab_end    = cell2mat(timeseries(:,21));
var_log_cons_end = cell2mat(timeseries(:,22));

% Model with fixed spillovers

[timeseries_fix,t_ss] = hand_fixS_n_llprof_eta(x_s);

dissim_fix      = cell2mat(timeseries_fix(:,1));
Q1toQ1_fix      = cell2mat(timeseries_fix(:,4));
Q1toQ4_fix      = cell2mat(timeseries_fix(:,5));
Q4toQ4_fix      = cell2mat(timeseries_fix(:,6));
rank_corr_fix   = cell2mat(timeseries_fix(:,7));
S_A_fix         = cell2mat(timeseries_fix(:,9));
S_B_fix         = cell2mat(timeseries_fix(:,10));
var_logw_k_fix = cell2mat(timeseries_fix(:,19));
dissim_ab_fix   = cell2mat(timeseries_fix(:,21));
var_log_cons_fix = cell2mat(timeseries_fix(:,22));

%% Figures
T=10;

fig=figure;
hold on
plot(1980:10:1980+(T)*10, S_A_end(t_ss-1:t_ss+T-1)-S_B_end(t_ss-1:t_ss+T-1),"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(1980:10:1980+(T)*10, S_A_fix(t_ss-1:t_ss+T-1)-S_B_fix(t_ss-1:t_ss+T-1),"--",'LineWidth',2,'Color', [0.8500    0.3250    0.0980] )
lgd=legend({'Endogenous Spillovers','Fixed Spillovers'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 2;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Spillover Gap')
xlabel('Time')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-dyn-wageshock-spill-gap.pdf','-dpdf','-r0')


fig=figure;
hold on
plot(1980:10:1980+(T)*10, var_logw_k_end(t_ss-1:t_ss+T-1),"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(1980:10:1980+(T)*10, var_logw_k_fix(t_ss-1:t_ss+T-1),"--",'LineWidth',2,'Color', [0.8500    0.3250    0.0980] )
lgd=legend({'Endogenous Spillovers','Fixed Spillovers'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 2;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Variance of Log Wages')
xlabel('Time')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-dyn-wageshock-varlogw.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(1980:10:1980+(T)*10, dissim_end(t_ss-1:t_ss+T-1),"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(1980:10:1980+(T)*10, dissim_fix(t_ss-1:t_ss+T-1),"--",'LineWidth',2,'Color', [0.8500    0.3250    0.0980] )
lgd=legend({'Endogenous Spillovers','Fixed Spillovers'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 2;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Dissimilarity by Income')
xlabel('Time')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-dyn-wageshock-diss.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(1980:10:1980+(T)*10, dissim_ab_end(t_ss-1:t_ss+T-1),"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(1980:10:1980+(T)*10, dissim_ab_fix(t_ss-1:t_ss+T-1),"--",'LineWidth',2,'Color', [0.8500    0.3250    0.0980] )
lgd=legend({'Endogenous Spillovers','Fixed Spillovers'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 2;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Dissimilarity by Productivity')
xlabel('Time')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-dyn-wageshock-diss-ab.pdf','-dpdf','-r0')


fig=figure;
hold on
plot(1980:10:1980+(T)*10, var_log_cons_end(t_ss-1:t_ss+T-1),"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(1980:10:1980+(T)*10, var_log_cons_fix(t_ss-1:t_ss+T-1),"--",'LineWidth',2,'Color', [0.8500    0.3250    0.0980] )
lgd=legend({'Endogenous Spillovers','Fixed Spillovers'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 2;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Variance of Log Consumption')
xlabel('Time')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-dyn-wageshock-varlogc.pdf','-dpdf','-r0')

fig=figure;
hold on
plot(1980:10:1980+T*10, rank_corr_end(t_ss-1:t_ss+T-1),"-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
plot(1980:10:1980+T*10, rank_corr_fix(t_ss-1:t_ss+T-1),"--",'LineWidth',2,'Color', [0.8500    0.3250    0.0980] )
lgd=legend({'Endogenous Spillovers','Fixed Spillovers'}, 'Location','northoutside','FontSize',12,'Box','off');
lgd.NumColumns = 2;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Rank-Rank')
xlabel('Time')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-dyn-wageshock-rank_corr.pdf','-dpdf','-r0')

fig=figure;
hold on
h1=plot(1980:10:1980+T*10, Q1toQ1_end(t_ss-1:t_ss+T-1),"o-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
h2=plot(1980:10:1980+T*10, Q1toQ1_fix(t_ss-1:t_ss+T-1),"o--",'LineWidth',2,'Color', [0.8500    0.3250    0.0980] )
h3=plot(1980:10:1980+T*10, Q1toQ4_end(t_ss-1:t_ss+T-1),"s-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
h4=plot(1980:10:1980+T*10, Q1toQ4_fix(t_ss-1:t_ss+T-1),"s--",'LineWidth',2,'Color',[0.8500    0.3250    0.0980])
h5=plot(1980:10:1980+T*10, Q4toQ4_end(t_ss-1:t_ss+T-1),"^-",'LineWidth',2,'Color', [0    0.4470    0.7410] )
h6=plot(1980:10:1980+T*10, Q4toQ4_fix(t_ss-1:t_ss+T-1),"^--",'LineWidth',2,'Color',[0.8500    0.3250    0.0980])
lgd=legend({'Q1 to Q1 Endog','Q1 to Q1 Fixed','Q1 to Q4 Endog','Q1 to Q4 Fixed','Q4 to Q4 Endog','Q4 to Q4 Fixed'}, 'Location','northoutside','FontSize',10,'Box','off');
lgd.NumColumns = 3;
set(gca,'XGrid','on','YGrid','on')
ax = gca;
ax.FontSize = 12; 
ylabel('Quartile Transitions')
xlabel('Time')
box on
hold off
set(fig,'Units','Inches');
pos = get(fig,'Position');
set(fig,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)])
print(fig,'figures/fig-dyn-wageshock-qtrans.pdf','-dpdf','-r0')

